// universalJsonLoader.js - Script autonome pour décompresser JSON
// Compatible avec tous les navigateurs modernes

(function() {
    'use strict'
    
    // Import dynamique de pako depuis CDN (fallback si pas déjà chargé)
    const loadPako = async () => {
        if (window.pako) return window.pako
        
        // Charger pako depuis CDN
        return new Promise((resolve, reject) => {
            const script = document.createElement('script')
            script.src = 'https://cdnjs.cloudflare.com/ajax/libs/pako/2.1.0/pako.min.js'
            script.onload = () => resolve(window.pako)
            script.onerror = () => reject(new Error('Impossible de charger pako'))
            document.head.appendChild(script)
        })
    }
    
    // Détecte le type de compression
    const detectCompressionType = (filename) => {
        const ext = filename.toLowerCase()
        
        if (ext.endsWith('.gz') || ext.endsWith('.gzip')) return 'gzip'
        if (ext.endsWith('.zip')) return 'zip'
        if (ext.endsWith('.br') || ext.endsWith('.brotli')) return 'brotli'
        if (ext.endsWith('.deflate')) return 'deflate'
        
        return 'none'
    }
    
    // Décompresse les données selon le type
    const decompressData = async (data, type, pako) => {
        switch (type) {
            case 'gzip':
                console.log('🗜️ Décompression GZIP...')
                return pako.ungzip(new Uint8Array(data), { to: 'string' })
                
            case 'deflate':
                console.log('🗜️ Décompression DEFLATE...')
                return pako.inflate(new Uint8Array(data), { to: 'string' })
                
            case 'zip':
                console.log('🗜️ Décompression ZIP...')
                throw new Error('Format ZIP non supporté dans cette version')
                
            case 'brotli':
                console.log('🗜️ Décompression BROTLI...')
                if ('DecompressionStream' in window) {
                    const stream = new Response(data).body.pipeThrough(new DecompressionStream('gzip'))
                    return await new Response(stream).text()
                } else {
                    throw new Error('Décompression Brotli non supportée par ce navigateur')
                }
                
            case 'none':
                console.log('📄 Pas de compression détectée')
                return new TextDecoder('utf-8').decode(data)
                
            default:
                throw new Error(`Type de compression non supporté: ${type}`)
        }
    }
    
    // Charge un fichier JSON (compressé ou non)
    const loadUniversalJSON = async (url) => {
        console.log(`📥 Chargement: ${url}`)
        
        const response = await fetch(url)
        
        if (!response.ok) {
            throw new Error(`HTTP ${response.status}: ${response.statusText}`)
        }
        
        const data = await response.arrayBuffer()
        const compressedSize = data.byteLength
        
        console.log(`📦 Fichier téléchargé: ${(compressedSize / 1024 / 1024).toFixed(1)}MB`)
        
        const compressionType = detectCompressionType(url)
        
        let decompressed
        if (compressionType === 'gzip' || compressionType === 'deflate') {
            const pako = await loadPako()
            decompressed = await decompressData(data, compressionType, pako)
        } else {
            decompressed = await decompressData(data, compressionType)
        }
        
        const decompressedSize = decompressed.length
        
        if (compressionType !== 'none') {
            const ratio = ((1 - compressedSize / decompressedSize) * 100)
            console.log(`📊 Décompression: ${(decompressedSize / 1024 / 1024).toFixed(1)}MB (${ratio.toFixed(1)}% économie)`)
        }
        
        const jsonData = JSON.parse(decompressed)
        console.log(`✅ JSON chargé avec succès`)
        
        return jsonData
    }
    
    // Fonction principale : charge avec fallback automatique
    const loadJSONWithFallback = async (baseName, extensions = ['json.gz', 'json.zip', 'json.br', 'json']) => {
        const isFileProtocol = window.location.protocol === 'file:'
        
        if (isFileProtocol) {
            console.warn('⚠️ Protocole file:// détecté - fallback vers chargement <script>')
            return await loadWithScriptTag(baseName)
        }
        
        const errors = []
        
        for (const ext of extensions) {
            const url = `${baseName}.${ext}`
            
            try {
                console.log(`🔄 Tentative: ${url}`)
                return await loadUniversalJSON(url)
                
            } catch (error) {
                const errorMsg = `${url}: ${error.message}`
                errors.push(errorMsg)
                console.warn(`⚠️ ${errorMsg}`)
            }
        }
        
        throw new Error(`Impossible de charger ${baseName} avec aucun format testé`)
    }
    
    // Fallback pour protocole file://
    const loadWithScriptTag = async (baseName) => {
        return new Promise((resolve, reject) => {
            const jsonUrl = `${baseName}.json`
            
            fetch(jsonUrl, { method: 'HEAD' })
                .then(response => {
                    if (response.ok) {
                        loadJsonWithScript(jsonUrl, baseName, resolve, reject)
                    } else {
                        reject(new Error(`Fichier ${jsonUrl} non trouvé. Utilisez un serveur HTTP local pour les fichiers compressés.`))
                    }
                })
                .catch(() => {
                    console.log(`🔄 Test d'existence échoué (CORS), tentative directe...`)
                    loadJsonWithScript(jsonUrl, baseName, resolve, reject)
                })
        })
    }
    
    const loadJsonWithScript = (jsonUrl, baseName, resolve, reject) => {
        const script = document.createElement('script')
        
        console.log(`📄 Chargement via <script>: ${jsonUrl}`)
        
        script.onload = () => {
            setTimeout(() => {
                const fileName = baseName.split('/').pop()
                
                let data = null
                const possibleKeys = [
                    fileName,
                    `RefFlux2_${fileName}`,
                    baseName,
                    Object.keys(window.sankey || {}).find(key => key.includes(fileName))
                ]
                
                for (const key of possibleKeys) {
                    if (window.sankey && window.sankey[key]) {
                        data = window.sankey[key]
                        console.log(`✅ Données trouvées avec la clé: ${key}`)
                        break
                    }
                }
                
                if (data) {
                    resolve(data)
                } else {
                    reject(new Error(`Données non trouvées dans window.sankey`))
                }
                
                document.head.removeChild(script)
            }, 100)
        }
        
        script.onerror = () => {
            document.head.removeChild(script)
            reject(new Error(`Erreur de syntaxe dans ${jsonUrl}. Utilisez un serveur HTTP local.`))
        }
        
        script.src = jsonUrl
        document.head.appendChild(script)
    }
    
    // Exposer les fonctions globalement
    window.loadCompressedJSON = loadJSONWithFallback
    window.loadUniversalJSON = loadUniversalJSON
    
    console.log('✅ universalJsonLoader.js chargé')
    
})();